package com.agilex.healthcare.mobilehealthplatform.web

import gov.va.vamf.scheduling.direct.web.XSSRequestFilter
import spock.lang.Specification
import spock.lang.Unroll

import javax.ws.rs.WebApplicationException
import javax.ws.rs.container.ContainerRequestContext
import javax.ws.rs.core.Cookie
import javax.ws.rs.core.MultivaluedHashMap
import javax.ws.rs.core.MultivaluedMap
import javax.ws.rs.core.UriInfo

class XssRequestFilterSpec extends Specification {
    private static String XSS_POSITIVE = "<script%20type='text/javascript'>alert('xss');</script>"

    def 'XSSRequestFilter does not change a valid request'() {
        given:
        def filter = new XSSRequestFilter()
        def actualPathParams = constructPathParams()
        def actualHeaders = constructHeaders()
        def actualCookies = constructCookies()

        when:
        filter.filter(constructRequestContext(actualPathParams, actualHeaders, actualCookies))

        then:
        def expectedHeaders = constructHeaders()
        actualHeaders.equalsIgnoreValueOrder(expectedHeaders)
        def expectedCookies = constructCookies()
        actualCookies == expectedCookies
        def expectedPathParams = constructPathParams()
        actualPathParams.equalsIgnoreValueOrder(expectedPathParams)
    }

    @Unroll
    def 'XSSRequestFilter throws an exception when a #testId has an XSS injection'() {
        given:
        def filter = new XSSRequestFilter()

        when:
        filter.filter(constructRequestContext(pathParams, headers, cookies))

        then:
        thrown WebApplicationException

        where:
        testId   | pathParams               | headers               | cookies
        'header' | constructPathParams()    | constructBadHeaders() | constructCookies()
        'param'  | constructBadPathParams() | constructHeaders()    | constructCookies()
        'cookie' | constructPathParams()    | constructHeaders()    | constructBadCookies()
    }

    private def constructCookies() {
        [
                new Cookie("chocolate chip","039fso0439j5g8","path","cookies.com",0),
                new Cookie("oatmeal","45f89heoirisfoisoisrj",null,"cookies.com",0),
                new Cookie("macaroon",null,"path","cookies.com",0)
        ].collectEntries { [it.name, it] }
    }

    private def constructBadCookies() {
        def cookies = constructCookies()
        cookies['oatmeal'] = new Cookie("oatmeal", XSS_POSITIVE,null,"cookies.com",0)
        return cookies
    }

    private def constructHeaders() {
        MultivaluedMap<String,String> map = new MultivaluedHashMap<>([
                header1: 'value1',
                header2: 'value2',
                header3: 'value3',
        ])
        map['header4'] = [ 'value4', 'value5', 'value6']

        return map
    }

    private def constructBadHeaders() {
        def headers = constructHeaders()
        headers.putSingle('header1', XSS_POSITIVE)
        return headers
    }

    private def constructPathParams(){
        def map = new MultivaluedHashMap<>([
                param1: 'value1',
                param2: 'value2',
                param3: 'value3'
        ])
        map['param4'] = [ 'value4', 'value5', 'value6' ]
        return map
    }

    private def constructBadPathParams(){
        def params = constructPathParams()
        params.putSingle('param1', XSS_POSITIVE)
        return params
    }

    private def constructRequestContext(pathParams, headers, cookies) {
        def uriInfo = Stub(UriInfo) {
            getPathParameters() >> pathParams
        }

        Stub(ContainerRequestContext) {
            getHeaders() >> headers
            getCookies() >> cookies
            getUriInfo() >> uriInfo
        }
    }
}
